/*
 * Decompiled with CFR 0.152.
 */
package ghidra.file.formats.squashfs;

import ghidra.app.util.bin.BinaryReader;
import ghidra.app.util.bin.ByteProvider;
import ghidra.file.formats.squashfs.SquashBasicFileInode;
import ghidra.file.formats.squashfs.SquashDirectoryTable;
import ghidra.file.formats.squashfs.SquashFileSystemFactory;
import ghidra.file.formats.squashfs.SquashFragment;
import ghidra.file.formats.squashfs.SquashFragmentTable;
import ghidra.file.formats.squashfs.SquashInode;
import ghidra.file.formats.squashfs.SquashInodeTable;
import ghidra.file.formats.squashfs.SquashSuperBlock;
import ghidra.file.formats.squashfs.SquashSymlinkInode;
import ghidra.file.formats.squashfs.SquashUtils;
import ghidra.file.formats.squashfs.SquashedFile;
import ghidra.formats.gfilesystem.AbstractFileSystem;
import ghidra.formats.gfilesystem.FSRLRoot;
import ghidra.formats.gfilesystem.FileSystemIndexHelper;
import ghidra.formats.gfilesystem.FileSystemService;
import ghidra.formats.gfilesystem.GFile;
import ghidra.formats.gfilesystem.GFileSystem;
import ghidra.formats.gfilesystem.annotations.FileSystemInfo;
import ghidra.formats.gfilesystem.fileinfo.FileAttributeType;
import ghidra.formats.gfilesystem.fileinfo.FileAttributes;
import ghidra.util.exception.CancelledException;
import ghidra.util.task.TaskMonitor;
import java.io.IOException;
import java.io.OutputStream;
import java.util.Arrays;
import java.util.Date;

@FileSystemInfo(type="squashfs", description="SquashFS", factory=SquashFileSystemFactory.class)
public class SquashFileSystem
extends AbstractFileSystem<SquashedFile> {
    private ByteProvider provider;
    private BinaryReader reader;
    private SquashSuperBlock superBlock;

    public SquashFileSystem(FSRLRoot fsFSRL, ByteProvider provider, FileSystemService fsService) {
        super(fsFSRL, fsService);
        this.fsIndex = new FileSystemIndexHelper((GFileSystem)this, fsFSRL);
        this.provider = provider;
        this.reader = new BinaryReader(provider, true);
    }

    public void mount(TaskMonitor monitor) throws IOException, CancelledException {
        monitor.setMessage("Opening " + SquashFileSystem.class.getSimpleName() + "...");
        this.superBlock = new SquashSuperBlock(this.reader);
        SquashFragmentTable fragmentTable = new SquashFragmentTable(this.reader, this.superBlock, monitor);
        SquashDirectoryTable directoryTable = new SquashDirectoryTable(this.reader, this.superBlock, fragmentTable, monitor);
        SquashInodeTable inodes = new SquashInodeTable(this.reader, this.superBlock, monitor);
        inodes.buildRelationships(monitor);
        directoryTable.assignInodes(inodes, monitor);
        SquashUtils.buildDirectoryStructure(fragmentTable, directoryTable, inodes, (FileSystemIndexHelper<SquashedFile>)this.fsIndex, monitor);
    }

    public ByteProvider getByteProvider(GFile file, TaskMonitor monitor) throws IOException, CancelledException {
        SquashedFile squashFile = (SquashedFile)this.fsIndex.getMetadata(file);
        long fileSize = -1L;
        if (squashFile != null) {
            fileSize = squashFile.getUncompressedSize();
        }
        return this.fsService.getDerivedByteProviderPush(this.provider.getFSRL(), file.getFSRL(), file.getName(), fileSize, os -> this.extractFileToStream(os, file, monitor), monitor);
    }

    public void extractFileToStream(OutputStream os, GFile file, TaskMonitor monitor) throws IOException, CancelledException {
        SquashedFile squashedFile = (SquashedFile)this.fsIndex.getMetadata(file = this.fsIndex.resolveSymlinks(file));
        if (squashedFile == null) {
            throw new IOException("Could not find SquashedFile associated with the symlink target");
        }
        SquashInode inode = squashedFile.getInode();
        if (!inode.isFile()) {
            throw new IOException("Inode is not a file");
        }
        SquashBasicFileInode fileInode = (SquashBasicFileInode)inode;
        long totalUncompressedBytes = 0L;
        monitor.initialize(fileInode.getFileSize());
        totalUncompressedBytes += (long)this.processFileBlocks(squashedFile, fileInode, os, monitor);
        if (squashedFile.hasFragment()) {
            totalUncompressedBytes += (long)this.processTailEnd(squashedFile, fileInode, os, monitor);
        }
        monitor.setProgress(totalUncompressedBytes);
    }

    private int processFileBlocks(SquashedFile squashedFile, SquashBasicFileInode fileInode, OutputStream os, TaskMonitor monitor) throws CancelledException, IOException {
        int[] blockSizes = fileInode.getBlockSizes();
        long location = fileInode.getStartBlockOffset();
        int blockUncompressedBytes = 0;
        for (int blockSizeHeader : blockSizes) {
            monitor.checkCancelled();
            monitor.setProgress((long)blockUncompressedBytes);
            boolean isCompressed = (blockSizeHeader & 0x1000000) == 0;
            long size = blockSizeHeader & 0xFEFFFFFF;
            if (size <= 0L) {
                size = this.superBlock.getBlockSize();
                os.write(new byte[(int)size]);
                blockUncompressedBytes = (int)((long)blockUncompressedBytes + size);
                continue;
            }
            this.reader.setPointerIndex(location);
            location += size;
            byte[] buffer = null;
            buffer = isCompressed ? SquashUtils.decompressBytes(this.reader, (int)size, this.superBlock.getCompressionType(), monitor) : this.reader.readNextByteArray((int)size);
            os.write(buffer);
            blockUncompressedBytes += buffer.length;
        }
        return blockUncompressedBytes;
    }

    private int processTailEnd(SquashedFile squashedFile, SquashBasicFileInode fileInode, OutputStream os, TaskMonitor monitor) throws CancelledException, IOException {
        SquashFragment fragment = squashedFile.getFragment();
        byte[] buffer = null;
        if (fragment.isCompressed()) {
            this.reader.setPointerIndex(fragment.getFragmentOffset());
            buffer = SquashUtils.decompressBytes(this.reader, (int)fragment.getFragmentSize(), this.superBlock.getCompressionType(), monitor);
            buffer = Arrays.copyOfRange(buffer, fileInode.getBlockOffset(), fileInode.getBlockOffset() + fileInode.getTailEndSize());
        } else {
            this.reader.setPointerIndex(fragment.getFragmentOffset() + (long)fileInode.getBlockOffset());
            buffer = this.reader.readNextByteArray(fileInode.getTailEndSize());
        }
        os.write(buffer);
        return buffer.length;
    }

    public boolean isClosed() {
        return this.provider == null;
    }

    public FileAttributes getFileAttributes(GFile file, TaskMonitor monitor) {
        FileAttributes result = new FileAttributes();
        SquashedFile squashedFile = (SquashedFile)this.fsIndex.getMetadata(file);
        if (squashedFile != null) {
            SquashInode inode = squashedFile.getInode();
            if (this.fsIndex.getRootDir().equals((Object)file)) {
                result.add("Compression used", (Object)this.superBlock.getCompressionTypeString());
                result.add("Block size", (Object)this.superBlock.getBlockSize());
                result.add("Inode count", (Object)this.superBlock.getInodeCount());
                result.add("Fragment count", (Object)this.superBlock.getTotalFragments());
                result.add("SquashFS version", (Object)this.superBlock.getVersionString());
                result.add(FileAttributeType.MODIFIED_DATE_ATTR, (Object)new Date(this.superBlock.getModTime()));
            } else {
                result.add(FileAttributeType.MODIFIED_DATE_ATTR, (Object)new Date(inode.getModTime()));
            }
            result.add(FileAttributeType.NAME_ATTR, (Object)squashedFile.getName());
            result.add(FileAttributeType.FSRL_ATTR, (Object)file.getFSRL());
            if (inode.isFile()) {
                SquashBasicFileInode fileInode = (SquashBasicFileInode)inode;
                result.add(FileAttributeType.SIZE_ATTR, (Object)squashedFile.getUncompressedSize());
                result.add(FileAttributeType.COMPRESSED_SIZE_ATTR, (Object)fileInode.getCompressedFileSize());
            } else if (inode.isSymLink()) {
                SquashSymlinkInode symLinkInode = (SquashSymlinkInode)inode;
                result.add(FileAttributeType.SYMLINK_DEST_ATTR, (Object)symLinkInode.getPath());
            }
        }
        return result;
    }

    public void close() throws IOException {
        this.refManager.onClose();
        this.fsIndex.clear();
        if (this.provider != null) {
            this.provider.close();
            this.provider = null;
        }
    }
}

